/**
 * Modularized AJAX Grid System
 * Uses snake_case and wrapped in a jQuery plugin function
 * Usage:
 * $("#my_grid").MY_AJAX_GRID({ source_file_path: '/employee/grid' });
 */

(function ($) {
    $.fn.MY_AJAX_GRID = function (user_options) {
        const grid_element = this;
        const result_div = $(this).attr('id');

        const default_options = {
            source_file_path: "#",
            action_buttons: "all",
        };

        const options = $.extend({}, default_options, user_options);

        if (!options.source_file_path || options.source_file_path === "#") {
            alert("The source path is not defined or invalid.");
            return;
        }

        if (!result_div) {
            alert("Grid Div ID is not defined.");
            return;
        }

        const create_grid_toolbar_html = () => {
            let html = `
                <div class="row grid-top-toolbar" id="grid_top_toolbar">
                    <div class="col-md-6">
                        <div class="grid_search_form_div mb-1">
                            <form method="post" action="#" id="grid_search_form" class="grid_search_form">
                                <table class="grid-search-box-table">
                                    <tr>
                                        <td>
                                            <div class="input-group">
                                                <input type="text" name="search_term" id="grid_search_term" class="form-control grid_search_term" placeholder="Search">
                                                <button type="submit" class="btn btn-primary grid_search_btn"><i class="fa fa-search"></i></button>
                                            </div>
                                        </td>
                                        <td>
                                            <select name="grid_limit" id="grid_limit" class="form-select grid_limit ms-2">
                                                <option value="10">10</option>
                                                <option value="25">25</option>
                                                <option value="50">50</option>
                                                <option value="100">100</option>
                                            </select>
                                        </td>
                                    </tr>
                                </table>
                            </form>
                        </div>
                    </div>`;

            if (options.action_buttons) {
                const buttons = options.action_buttons.includes("all") ? ["copy", "csv", "excel", "pdf", "print"] : options.action_buttons.split(',');
                html += `
                    <div class="col-md-6">
                        <div class="btn-group mb-1 float-end">
                            ${buttons.includes("copy") ? '<button class="btn btn-secondary grid_btn_copy"><i class="fas fa-copy"></i> Copy</button>' : ""}
                            ${buttons.includes("csv") ? '<button class="btn btn-secondary grid_btn_csv"><i class="fas fa-file-csv"></i> CSV</button>' : ""}
                            ${buttons.includes("excel") ? '<button class="btn btn-secondary grid_btn_excel"><i class="fas fa-file-excel"></i> Excel</button>' : ""}
                            ${buttons.includes("pdf") ? '<button class="btn btn-secondary grid_btn_pdf"><i class="fas fa-file-pdf"></i> PDF</button>' : ""}
                            ${buttons.includes("print") ? '<button class="btn btn-secondary grid_btn_print"><i class="fas fa-print"></i> Print</button>' : ""}
                        </div>
                    </div>`;
            }

            html += `</div>`;
            return html;
        };

        const create_grid_data_div = () => `<div id="grid_result" class="row grid_result"></div>`;

        const is_json_string = str => {
            try {
                JSON.parse(str);
                return true;
            } catch (e) {
                return false;
            }
        };

        const show_loader = (div) => {
            $(div).html(`
                <div class="text-center w-100 py-4">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <div class="mt-2">Loading...</div>
                </div>`);
        };

        const hide_loader = (div) => {
            $(`${div} #ajax-loader`).hide();
        };

        const load_grid = (page = 1, load_more = false) => {
            const limit = $(`#${result_div} .grid_limit`).val() || "";
            const search_term = $(`#${result_div} .grid_search_term`).val() || "";
            const data = `action=load&page=${page}&limit=${limit}&search_term=${search_term}`;

            const loader_target = load_more ? `#${result_div} .grid_load_more_loader` : `#${result_div} .grid_result`;
            show_loader(loader_target);

            $.ajax({
                type: "GET",
                url: options.source_file_path,
                data: data,
                cache: false,
                success: function (response) {
                    if (load_more) {
                        $(`#${result_div} .grid_result`).append(response);
                    } else {
                        $(`#${result_div} .grid_result`).html(response);
                    }

                    if (is_json_string(response)) {
                        const json = JSON.parse(response);
                        if (json.error) {
                            $(`#${result_div}`).html(json.error);
                            hide_loader(`#${result_div} .grid_result`);
                        }
                    }

                    hide_loader(`#${result_div} .grid_result`);
                },
                error: function (xhr, status, error) {
                    alert(`${status} - Error: ${error}`);
                    hide_loader(`#${result_div} .grid_result`);
                }
            });
        };

        // Append toolbar and result div
        $(this).html(create_grid_toolbar_html() + create_grid_data_div());

        // Load grid on init
        load_grid();

        // Event listeners
        $(`#${result_div}`).on("change", ".grid_limit", () => load_grid());
        $(`#${result_div}`).on("keydown", ".grid_search_form", function (e) {
            if (e.keyCode === 13) {
                e.preventDefault();
                load_grid();
            }
        });
        $(`#${result_div}`).on("click", ".grid_search_btn", function (e) {
            e.preventDefault();
            load_grid();
        });

        // Exports (hooks, optional - to be implemented separately)
        $(`#${result_div}`).on("click", "button.grid_btn_excel", e => { e.preventDefault(); export_to_excel(result_div); });
        $(`#${result_div}`).on("click", "button.grid_btn_csv", e => { e.preventDefault(); export_to_csv(result_div); });
        $(`#${result_div}`).on("click", "button.grid_btn_pdf", e => { e.preventDefault(); export_to_pdf(result_div); });
        $(`#${result_div}`).on("click", "button.grid_btn_copy", e => { e.preventDefault(); copy_grid_table(result_div); });
        $(`#${result_div}`).on("click", "button.grid_btn_print", e => { e.preventDefault(); print_grid_table(result_div); });

        // Public API
        return {
            load_grid: load_grid
        };
    };
})(jQuery);






